import { Card, List, Button, Tag, Space, Segmented } from "antd";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../api/client";
import { useState } from "react";
import { useNotificationStore } from "../store/notificationStore";

const NotificationsPage = () => {
  const queryClient = useQueryClient();
  const [filter, setFilter] = useState<"all" | "unread">("all");
  const { unreadCount, setUnreadCount } = useNotificationStore();
  const { data, isLoading } = useQuery({
    queryKey: ["notifications", filter],
    queryFn: async () => {
      const response = await api.get("/notifications", {
        params: { page: 1, page_size: 50, unread: filter === "unread" ? true : undefined }
      });
      return response.data.data.notifications || [];
    }
  });

  const markAll = async () => {
    await api.post("/notifications/mark-all-read");
    setUnreadCount(0);
    queryClient.invalidateQueries({ queryKey: ["notifications"] });
  };

  const markRead = async (id: number) => {
    await api.post("/notifications/mark-read", { ids: [id] });
    setUnreadCount(Math.max(0, unreadCount - 1));
    queryClient.invalidateQueries({ queryKey: ["notifications"] });
  };

  return (
    <Card
      className="panel"
      title="Notification Center"
      extra={
        <Space>
          <Segmented
            options={[
              { label: "All", value: "all" },
              { label: "Unread", value: "unread" }
            ]}
            value={filter}
            onChange={(v) => setFilter(v as any)}
          />
          <Button onClick={markAll}>Mark all read</Button>
        </Space>
      }
    >
      <List
        loading={isLoading}
        dataSource={data || []}
        renderItem={(item: any) => (
          <List.Item
            actions={[
              item.read_at ? (
                <Tag color="green">Read</Tag>
              ) : (
                <Button size="small" onClick={() => markRead(item.id)}>
                  Mark read
                </Button>
              )
            ]}
          >
            <List.Item.Meta title={item.title} description={item.message} />
            <div style={{ color: "#94a3b8" }}>{item.created_at}</div>
          </List.Item>
        )}
      />
    </Card>
  );
};

export default NotificationsPage;
