import { Card, Descriptions, Row, Col, Select, Tabs, Table, Space, Typography } from "antd";
import { useEffect, useMemo, useRef, useState } from "react";
import { useAuthStore } from "../store/authStore";
import { useQuery } from "@tanstack/react-query";
import ReactECharts from "echarts-for-react";
import api from "../api/client";
import PlanBadge from "../components/PlanBadge";

const { Text } = Typography;

const ProfilePage = () => {
  const user = useAuthStore((state) => state.user);
  const [membershipId, setMembershipId] = useState<string>("");
  const [year, setYear] = useState<number>(new Date().getFullYear());
  const didInit = useRef(false);

  const { data: sessions } = useQuery({
    queryKey: ["my-sessions"],
    queryFn: async () => {
      const response = await api.get("/sessions/my");
      return response.data.data;
    }
  });

  const membershipsQuery = useQuery({
    queryKey: ["memberships-selector", user?.id],
    enabled: Boolean(user),
    queryFn: async () => {
      const response = await api.get("/memberships/selector");
      return response.data.data;
    }
  });

  const selectedMembership = useMemo(() => {
    if (!membershipId || membershipId === "all") return null;
    return (membershipsQuery.data || []).find((m: any) => m.member_id === membershipId) || null;
  }, [membershipId, membershipsQuery.data]);

  const activeMembershipsQuery = useQuery({
    queryKey: ["memberships-active-profile", user?.id],
    enabled: Boolean(user),
    queryFn: async () => (await api.get("/memberships/active")).data.data
  });

  useEffect(() => {
    didInit.current = false;
    setMembershipId("");
  }, [user?.id]);

  useEffect(() => {
    if (didInit.current) return;
    if (!user) return;

    const all = membershipsQuery.data || [];
    const active = activeMembershipsQuery.data || [];
    const activeFirst = active[0]?.member_id;
    const anyFirst = all.find((m: any) => m.status === "active")?.member_id || all[0]?.member_id;

    if (user.access_level === "administrator") {
      setMembershipId(activeFirst || anyFirst || "all");
    } else {
      setMembershipId(activeFirst || anyFirst || "");
    }
    didInit.current = true;
  }, [user, membershipsQuery.data, activeMembershipsQuery.data]);

  const yearsQuery = useQuery({
    queryKey: ["profile-years", membershipId],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-years", { params: { membership_id: membershipId } });
      return response.data.data.years || [];
    }
  });

  const availableYears = useMemo(() => yearsQuery.data || [], [yearsQuery.data]);

  useEffect(() => {
    if (!membershipId) return;
    if (!availableYears.length) return;
    const oldest = Math.min(...availableYears);
    setYear(oldest);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [membershipId, availableYears.join(",")]);

  const { data: summary } = useQuery({
    queryKey: ["profile-summary", membershipId],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-summary", { params: { membership_id: membershipId } });
      return response.data.data;
    }
  });

  const { data: monthly } = useQuery({
    queryKey: ["profile-monthly", membershipId, year],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-monthly", { params: { membership_id: membershipId, year } });
      return response.data.data;
    }
  });

  const { data: requestsSummary } = useQuery({
    queryKey: ["profile-requests-summary", membershipId],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-requests-summary", { params: { membership_id: membershipId } });
      return response.data.data;
    }
  });

  const { data: savingsVsLoan } = useQuery({
    queryKey: ["profile-savings-vs-loan", membershipId],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-savings-vs-loan", { params: { membership_id: membershipId } });
      return response.data.data;
    }
  });

  const { data: creditHistory } = useQuery({
    queryKey: ["profile-history-credits", membershipId],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-history/credits", { params: { membership_id: membershipId } });
      return response.data.data.items || [];
    }
  });

  const { data: debitHistory } = useQuery({
    queryKey: ["profile-history-debits", membershipId],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-history/debits", { params: { membership_id: membershipId } });
      return response.data.data.items || [];
    }
  });

  const { data: requestHistory } = useQuery({
    queryKey: ["profile-history-requests", membershipId],
    enabled: Boolean(membershipId),
    queryFn: async () => {
      const response = await api.get("/reports/profile-history/requests", { params: { membership_id: membershipId } });
      return response.data.data;
    }
  });

  const monthlyOption = {
    tooltip: { trigger: "axis" },
    legend: { textStyle: { color: "#e2e8f0" } },
    xAxis: { type: "category", data: monthly?.months || [], axisLabel: { color: "#94a3b8" } },
    yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
    series: [
      { name: "Savings", type: "bar", data: monthly?.savings || [] },
      { name: "Charity", type: "bar", data: monthly?.charity || [] },
      { name: "Loans", type: "bar", data: monthly?.loans || [] },
      { name: "EMI", type: "bar", data: monthly?.emi || [] }
    ]
  };

  const requestsOption = {
    tooltip: {},
    legend: { textStyle: { color: "#e2e8f0" } },
    xAxis: {
      type: "category",
      data: ["Loan", "Charity", "Release"],
      axisLabel: { color: "#94a3b8" }
    },
    yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
    series: [
      {
        name: "Requests",
        type: "bar",
        data: [
          (Object.values(requestsSummary?.loan_requests || {}) as any[]).map(Number).reduce((a, b) => a + b, 0),
          (Object.values(requestsSummary?.charity_requests || {}) as any[]).map(Number).reduce((a, b) => a + b, 0),
          (Object.values(requestsSummary?.release_requests || {}) as any[]).map(Number).reduce((a, b) => a + b, 0)
        ]
      }
    ]
  };

  return (
    <Row gutter={[16, 16]}>
      <Col xs={24} md={8}>
        <Card className="panel" title="Profile">
          <Descriptions column={1} bordered size="small">
            <Descriptions.Item label="Full Name">{user?.full_name}</Descriptions.Item>
            <Descriptions.Item label="Username">{user?.username}</Descriptions.Item>
            <Descriptions.Item label="Email">{user?.email}</Descriptions.Item>
            <Descriptions.Item label="Role">{user?.access_level}</Descriptions.Item>
          </Descriptions>
        </Card>
        <Card className="panel" title="Membership" style={{ marginTop: 16 }}>
          <Select
            value={membershipId || undefined}
            onChange={(v) => {
              setMembershipId(v);
            }}
            style={{ width: "100%" }}
            showSearch
            optionFilterProp="label"
            placeholder="Select membership"
            options={[
              ...(user?.access_level === "administrator"
                ? [{ label: "All Memberships (System)", value: "all" }]
                : [{ label: "All My Memberships", value: "all" }]),
              ...(membershipsQuery.data || []).map((m: any) => ({
                label:
                  user?.access_level === "administrator"
                    ? `${m.member_id} - ${m.member_short_name} (${m.user_full_name || "User"})`
                    : `${m.member_id} - ${m.member_short_name}`,
                value: m.member_id
              }))
            ]}
          />
          {!membershipsQuery.isFetching && !membershipsQuery.data?.length && (
            <div style={{ marginTop: 10 }}>
              <Text type="secondary">No memberships found.</Text>
            </div>
          )}
          {selectedMembership?.plan_type_name && (
            <div style={{ marginTop: 10, display: "flex", alignItems: "center", gap: 10 }}>
              <PlanBadge planTypeName={selectedMembership.plan_type_name} size={40} />
              <div>
                <div style={{ fontWeight: 600 }}>{selectedMembership.plan_type_name}</div>
                <div style={{ color: "var(--app-muted)", fontSize: 12 }}>{selectedMembership.member_id}</div>
              </div>
            </div>
          )}
        </Card>
        <Card className="panel" title="Summary" style={{ marginTop: 16 }}>
          <Descriptions column={1} bordered size="small">
            <Descriptions.Item label="Total Savings">{summary?.total_savings || 0}</Descriptions.Item>
            <Descriptions.Item label="Total Charity">{summary?.total_charity || 0}</Descriptions.Item>
            <Descriptions.Item label="Total Borrowed">{summary?.total_borrowed || 0}</Descriptions.Item>
            <Descriptions.Item label="Outstanding">{summary?.outstanding || 0}</Descriptions.Item>
          </Descriptions>
        </Card>
      </Col>
      <Col xs={24} md={16}>
        <Card
          className="panel"
          title="Monthly Trends"
          extra={
            <Select
              value={year}
              onChange={setYear}
              style={{ width: 120 }}
              placeholder="Year"
              options={(availableYears.length ? availableYears : [new Date().getFullYear()]).map((y: number) => ({
                label: String(y),
                value: y
              }))}
            />
          }
        >
          <ReactECharts option={monthlyOption} style={{ height: 300 }} />
        </Card>
        <Row gutter={[16, 16]} style={{ marginTop: 16 }}>
          <Col xs={24} md={12}>
            <Card className="panel" title="Requests Summary">
              <ReactECharts option={requestsOption} style={{ height: 220 }} />
            </Card>
          </Col>
          <Col xs={24} md={12}>
            <Card className="panel" title="Savings vs Loans">
              <ReactECharts
                option={{
                  tooltip: { trigger: "item" },
                  legend: { textStyle: { color: "#e2e8f0" } },
                  series: [
                    {
                      type: "pie",
                      radius: ["35%", "70%"],
                      data: (savingsVsLoan?.labels || []).map((label: string, index: number) => ({
                        name: label,
                        value: savingsVsLoan?.values?.[index] || 0
                      }))
                    }
                  ]
                }}
                style={{ height: 220 }}
              />
            </Card>
          </Col>
        </Row>
      </Col>
      <Col xs={24}>
        <Card className="panel" title="History">
          <Tabs
            items={[
              {
                key: "credits",
                label: "Credits",
                children: (
                  <Table
                    rowKey="id"
                    dataSource={creditHistory || []}
                    columns={[
                      { title: "Transaction ID", dataIndex: "transaction_id" },
                      { title: "Member ID", dataIndex: "member_id" },
                      { title: "Type", dataIndex: "transaction_type" },
                      { title: "Total", dataIndex: "total_amount" },
                      { title: "Date", dataIndex: "transaction_date" }
                    ]}
                  />
                )
              },
              {
                key: "debits",
                label: "Debits",
                children: (
                  <Table
                    rowKey="id"
                    dataSource={debitHistory || []}
                    columns={[
                      { title: "Transaction ID", dataIndex: "transaction_id" },
                      { title: "Member ID", dataIndex: "member_id" },
                      { title: "Type", dataIndex: "transaction_type" },
                      { title: "Amount", dataIndex: "amount" },
                      { title: "Date", dataIndex: "transaction_date" }
                    ]}
                  />
                )
              },
              {
                key: "requests",
                label: "Requests",
                children: (
                  <Row gutter={[16, 16]}>
                    <Col xs={24} md={8}>
                      <Table
                        rowKey="id"
                        dataSource={requestHistory?.loan_requests || []}
                        columns={[
                          { title: "Request ID", dataIndex: "request_id" },
                          { title: "Amount", dataIndex: "amount" },
                          { title: "Status", dataIndex: "status" }
                        ]}
                        pagination={false}
                      />
                    </Col>
                    <Col xs={24} md={8}>
                      <Table
                        rowKey="id"
                        dataSource={requestHistory?.charity_requests || []}
                        columns={[
                          { title: "Charity ID", dataIndex: "charity_id" },
                          { title: "Amount", dataIndex: "amount" },
                          { title: "Status", dataIndex: "status" }
                        ]}
                        pagination={false}
                      />
                    </Col>
                    <Col xs={24} md={8}>
                      <Table
                        rowKey="id"
                        dataSource={requestHistory?.release_requests || []}
                        columns={[
                          { title: "Release ID", dataIndex: "release_request_id" },
                          { title: "Amount", dataIndex: "release_amount" },
                          { title: "Status", dataIndex: "status" }
                        ]}
                        pagination={false}
                      />
                    </Col>
                  </Row>
                )
              },
              {
                key: "sessions",
                label: "Sessions",
                children: (
                  <div>
                    {sessions?.map((session: any) => (
                      <div key={session.login_time} style={{ marginBottom: 12 }}>
                        <strong>{session.device_type}</strong> - {session.ip_address}
                        <div style={{ color: "#94a3b8" }}>{session.login_time}</div>
                      </div>
                    ))}
                  </div>
                )
              }
            ]}
          />
        </Card>
      </Col>
    </Row>
  );
};

export default ProfilePage;
