import { useMemo, useState } from "react";
import { Card, Row, Col, Statistic, Segmented, Table, Input, Space, Tabs, Button } from "antd";
import { useQuery } from "@tanstack/react-query";
import ReactECharts from "echarts-for-react";
import api from "../../api/client";
import { downloadCsv } from "../../utils/exportCsv";

const ActivityAdminPage = () => {
  const [period, setPeriod] = useState<"today" | "week" | "month" | "all">("today");
  const [search, setSearch] = useState("");

  const { data: stats, isLoading: statsLoading } = useQuery({
    queryKey: ["admin-activity-stats", period],
    queryFn: async () => {
      const response = await api.get("/admin/activity-stats", { params: { period } });
      return response.data.data;
    }
  });

  const { data: sessions, isLoading: sessionsLoading } = useQuery({
    queryKey: ["admin-sessions"],
    queryFn: async () => {
      const response = await api.get("/admin/sessions");
      return response.data.data;
    }
  });

  const filteredSessions = useMemo(() => {
    const q = search.trim().toLowerCase();
    if (!q) return sessions || [];
    return (sessions || []).filter((s: any) => {
      return (
        String(s.session_id || "").toLowerCase().includes(q) ||
        String(s.user_id || "").toLowerCase().includes(q) ||
        String(s.ip_address || "").toLowerCase().includes(q) ||
        String(s.browser || "").toLowerCase().includes(q) ||
        String(s.os || "").toLowerCase().includes(q) ||
        String(s.device_type || "").toLowerCase().includes(q) ||
        String(s.country || "").toLowerCase().includes(q) ||
        String(s.city || "").toLowerCase().includes(q)
      );
    });
  }, [sessions, search]);

  const deviceOption = useMemo(() => {
    const deviceStats = stats?.device_stats || {};
    const items = Object.entries(deviceStats).map(([name, value]) => ({ name, value }));
    return {
      tooltip: { trigger: "item" },
      legend: { textStyle: { color: "#e2e8f0" } },
      series: [{ type: "pie", radius: ["35%", "70%"], data: items }]
    };
  }, [stats]);

  const browserOption = useMemo(() => {
    const browserStats = stats?.browser_stats || {};
    const items = Object.entries(browserStats).map(([name, value]) => ({ name, value }));
    return {
      tooltip: { trigger: "item" },
      legend: { textStyle: { color: "#e2e8f0" } },
      series: [{ type: "pie", radius: ["35%", "70%"], data: items }]
    };
  }, [stats]);

  const osOption = useMemo(() => {
    const osStats = stats?.os_stats || {};
    const items = Object.entries(osStats).map(([name, value]) => ({ name, value }));
    return {
      tooltip: { trigger: "item" },
      legend: { textStyle: { color: "#e2e8f0" } },
      series: [{ type: "pie", radius: ["35%", "70%"], data: items }]
    };
  }, [stats]);

  return (
    <Card
      className="panel"
      title="Activity Dashboard"
      extra={
        <Segmented
          options={[
            { label: "Today", value: "today" },
            { label: "Week", value: "week" },
            { label: "Month", value: "month" },
            { label: "All", value: "all" }
          ]}
          value={period}
          onChange={(value) => setPeriod(value as any)}
        />
      }
    >
      <Tabs
        items={[
          {
            key: "stats",
            label: "Stats",
            children: (
              <Row gutter={[16, 16]}>
                <Col xs={24} md={6}>
                  <Card className="panel" loading={statsLoading}>
                    <Statistic title="Total Sessions" value={stats?.total_sessions || 0} />
                  </Card>
                </Col>
                <Col xs={24} md={6}>
                  <Card className="panel" loading={statsLoading}>
                    <Statistic title="Unique Users" value={stats?.unique_users || 0} />
                  </Card>
                </Col>
                <Col xs={24} md={6}>
                  <Card className="panel" loading={statsLoading}>
                    <Statistic title="Total Time (min)" value={stats?.total_time_minutes || 0} />
                  </Card>
                </Col>
                <Col xs={24} md={6}>
                  <Card className="panel" loading={statsLoading}>
                    <Statistic title="Avg Session (min)" value={stats?.avg_session_time_minutes || 0} />
                  </Card>
                </Col>

                <Col xs={24} md={8}>
                  <Card className="panel" title="Devices" loading={statsLoading}>
                    <ReactECharts option={deviceOption} style={{ height: 260 }} />
                  </Card>
                </Col>
                <Col xs={24} md={8}>
                  <Card className="panel" title="Browsers" loading={statsLoading}>
                    <ReactECharts option={browserOption} style={{ height: 260 }} />
                  </Card>
                </Col>
                <Col xs={24} md={8}>
                  <Card className="panel" title="OS" loading={statsLoading}>
                    <ReactECharts option={osOption} style={{ height: 260 }} />
                  </Card>
                </Col>
              </Row>
            )
          },
          {
            key: "sessions",
            label: "Sessions",
            children: (
              <>
                <Space style={{ marginBottom: 12 }}>
                  <Input.Search placeholder="Search sessions" allowClear onChange={(e) => setSearch(e.target.value)} />
                  <Button
                    onClick={() =>
                      downloadCsv("sessions.csv", filteredSessions, [
                        { title: "Session ID", dataIndex: "session_id" },
                        { title: "User ID", dataIndex: "user_id" },
                        { title: "IP", dataIndex: "ip_address" },
                        { title: "Device", dataIndex: "device_type" },
                        { title: "Browser", dataIndex: "browser" },
                        { title: "OS", dataIndex: "os" },
                        { title: "Country", dataIndex: "country" },
                        { title: "City", dataIndex: "city" },
                        { title: "Login", dataIndex: "login_time" },
                        { title: "Logout", dataIndex: "logout_time" },
                        { title: "Active", dataIndex: "is_active" }
                      ])
                    }
                  >
                    Export CSV
                  </Button>
                </Space>

                <Table
                  loading={sessionsLoading}
                  dataSource={filteredSessions}
                  rowKey="session_id"
                  pagination={{ pageSize: 20 }}
                  columns={[
                    { title: "Session", dataIndex: "session_id" },
                    { title: "User", dataIndex: "user_id" },
                    { title: "IP", dataIndex: "ip_address" },
                    { title: "Device", dataIndex: "device_type" },
                    { title: "Browser", dataIndex: "browser" },
                    { title: "OS", dataIndex: "os" },
                    { title: "Country", dataIndex: "country" },
                    { title: "City", dataIndex: "city" },
                    { title: "Login", dataIndex: "login_time" },
                    { title: "Logout", dataIndex: "logout_time" }
                  ]}
                />
              </>
            )
          }
        ]}
      />
    </Card>
  );
};

export default ActivityAdminPage;
