import { useMemo, useState } from "react";
import { Card, Table, Button, Drawer, Form, Input, Select, message, Space } from "antd";
import { PlusOutlined } from "@ant-design/icons";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../../api/client";
import { downloadCsv } from "../../utils/exportCsv";

const CashiersAdminPage = () => {
  const queryClient = useQueryClient();
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [form] = Form.useForm();
  const [search, setSearch] = useState("");
  const [statusFilter, setStatusFilter] = useState<"active" | "inactive" | undefined>();

  const { data, isLoading } = useQuery({
    queryKey: ["cashiers"],
    queryFn: async () => {
      const response = await api.get("/cashiers");
      return response.data.data;
    }
  });

  const { data: users } = useQuery({
    queryKey: ["admin-users"],
    queryFn: async () => {
      const response = await api.get("/admin/users");
      return response.data.data;
    }
  });

  const userOptions = useMemo(() => {
    const approvedUsers = (users || []).filter((u: any) => u.account_status === "Approved");
    const options = approvedUsers.map((u: any) => ({
      label: `${u.full_name} (${u.email})`,
      value: u.id
    }));
    if (editing?.user_id && !options.find((o: any) => o.value === editing.user_id)) {
      const current = (users || []).find((u: any) => u.id === editing.user_id);
      if (current) {
        options.unshift({
          label: `${current.full_name} (${current.email})`,
          value: current.id
        });
      }
    }
    return options;
  }, [users, editing]);

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    form.setFieldsValue({
      name: record.name,
      status: record.status,
      user_id: record.user_id
    });
  };

  const onCreate = () => {
    setEditing(null);
    form.resetFields();
    setOpen(true);
  };

  const onSave = async (values: any) => {
    try {
      if (editing) {
        await api.put(`/cashiers/${editing.id}`, values);
        message.success("Cashier updated");
      } else {
        await api.post("/cashiers", values);
        message.success("Cashier created");
      }
      setOpen(false);
      setEditing(null);
      form.resetFields();
      queryClient.invalidateQueries({ queryKey: ["cashiers"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  const onDelete = async (record: any) => {
    try {
      await api.delete(`/cashiers/${record.id}`);
      message.success("Cashier deleted");
      queryClient.invalidateQueries({ queryKey: ["cashiers"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Delete failed");
    }
  };

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    return (data || []).filter((row: any) => {
      if (statusFilter && row.status !== statusFilter) return false;
      if (!q) return true;
      return (
        String(row.name || "").toLowerCase().includes(q) ||
        String(row.status || "").toLowerCase().includes(q) ||
        String(row.user_id || "").toLowerCase().includes(q)
      );
    });
  }, [data, search, statusFilter]);

  return (
    <Card
      className="panel"
      title="Cashiers"
      extra={
        <Space>
          <Select
            allowClear
            placeholder="Status"
            style={{ width: 140 }}
            value={statusFilter}
            onChange={(v) => setStatusFilter(v)}
            options={[
              { label: "Active", value: "active" },
              { label: "Inactive", value: "inactive" }
            ]}
          />
          <Input.Search placeholder="Search" allowClear onChange={(e) => setSearch(e.target.value)} style={{ width: 220 }} />
          <Button onClick={() => downloadCsv("cashiers.csv", filtered, [
            { title: "Name", dataIndex: "name" },
            { title: "Status", dataIndex: "status" },
            { title: "User ID", dataIndex: "user_id" }
          ])}>Export CSV</Button>
          <Button type="primary" icon={<PlusOutlined />} onClick={onCreate}>
            New Cashier
          </Button>
        </Space>
      }
    >
      <Table
        loading={isLoading}
        dataSource={filtered}
        rowKey="id"
        columns={[
          { title: "Name", dataIndex: "name" },
          { title: "Status", dataIndex: "status" },
          { title: "User ID", dataIndex: "user_id" },
          {
            title: "Actions",
            render: (_: any, record: any) => (
              <Space>
                <Button size="small" onClick={() => onEdit(record)}>
                  Edit
                </Button>
                <Button size="small" danger onClick={() => onDelete(record)}>
                  Delete
                </Button>
              </Space>
            )
          }
        ]}
      />

      <Drawer
        title={editing ? "Edit Cashier" : "Create Cashier"}
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={420}
      >
        <Form form={form} layout="vertical" onFinish={onSave}>
          <Form.Item name="name" label="Name" rules={[{ required: true }]}>
            <Input />
          </Form.Item>
          <Form.Item name="status" label="Status" rules={[{ required: true }]}>
            <Select
              options={[
                { label: "Active", value: "active" },
                { label: "Inactive", value: "inactive" }
              ]}
            />
          </Form.Item>
          <Form.Item name="user_id" label="User">
            <Select
              allowClear
              options={userOptions}
            />
          </Form.Item>
          <Button type="primary" htmlType="submit" block>
            Save
          </Button>
        </Form>
      </Drawer>
    </Card>
  );
};

export default CashiersAdminPage;
