import { useMemo, useState } from "react";
import { Card, Table, Button, Drawer, Form, Input, Select, message, Space } from "antd";
import { PlusOutlined } from "@ant-design/icons";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../../api/client";
import { useAuthStore } from "../../store/authStore";
import { downloadCsv } from "../../utils/exportCsv";

const CharityTypesAdminPage = () => {
  const queryClient = useQueryClient();
  const { user } = useAuthStore();
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [form] = Form.useForm();
  const [search, setSearch] = useState("");
  const [statusFilter, setStatusFilter] = useState<"active" | "inactive" | undefined>();

  const { data, isLoading } = useQuery({
    queryKey: ["charity-types"],
    queryFn: async () => {
      const response = await api.get("/charity-types");
      return response.data.data;
    }
  });

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    form.setFieldsValue({
      charity_type: record.charity_type,
      status: record.status
    });
  };

  const onCreate = () => {
    setEditing(null);
    form.resetFields();
    setOpen(true);
  };

  const onSave = async (values: any) => {
    try {
      if (editing) {
        await api.put(`/charity-types/${editing.id}`, values);
        message.success("Charity type updated");
      } else {
        await api.post("/charity-types", values);
        message.success("Charity type created");
      }
      setOpen(false);
      setEditing(null);
      form.resetFields();
      queryClient.invalidateQueries({ queryKey: ["charity-types"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  const onDelete = async (record: any) => {
    try {
      await api.delete(`/charity-types/${record.id}`);
      message.success("Charity type deleted");
      queryClient.invalidateQueries({ queryKey: ["charity-types"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Delete failed");
    }
  };

  const columns = [
    { title: "Type", dataIndex: "charity_type" },
    { title: "Status", dataIndex: "status" }
  ];

  if (user?.access_level === "administrator") {
    columns.push({
      title: "Actions",
      render: (_: any, record: any) => (
        <Space>
          <Button size="small" onClick={() => onEdit(record)}>
            Edit
          </Button>
          <Button size="small" danger onClick={() => onDelete(record)}>
            Delete
          </Button>
        </Space>
      )
    } as any);
  }

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    return (data || []).filter((row: any) => {
      if (statusFilter && row.status !== statusFilter) return false;
      if (!q) return true;
      return String(row.charity_type || "").toLowerCase().includes(q);
    });
  }, [data, search, statusFilter]);

  return (
    <Card
      className="panel"
      title="Charity Types"
      extra={
        <Space>
          <Select
            allowClear
            placeholder="Status"
            style={{ width: 160 }}
            value={statusFilter}
            onChange={(v) => setStatusFilter(v)}
            options={[
              { label: "Active", value: "active" },
              { label: "Inactive", value: "inactive" }
            ]}
          />
          <Input.Search placeholder="Search type" allowClear onChange={(e) => setSearch(e.target.value)} style={{ width: 220 }} />
          <Button onClick={() => downloadCsv("charity_types.csv", filtered, columns as any)}>Export CSV</Button>
          {user?.access_level === "administrator" ? (
            <Button type="primary" icon={<PlusOutlined />} onClick={onCreate}>
              New Charity Type
            </Button>
          ) : null}
        </Space>
      }
    >
      <Table loading={isLoading} dataSource={filtered} rowKey="id" columns={columns} />

      <Drawer
        title={editing ? "Edit Charity Type" : "Create Charity Type"}
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={420}
      >
        <Form form={form} layout="vertical" onFinish={onSave}>
          <Form.Item name="charity_type" label="Type" rules={[{ required: true }]}>
            <Input />
          </Form.Item>
          <Form.Item name="status" label="Status" rules={[{ required: true }]}>
            <Select
              options={[
                { label: "Active", value: "active" },
                { label: "Inactive", value: "inactive" }
              ]}
            />
          </Form.Item>
          <Button type="primary" htmlType="submit" block>
            Save
          </Button>
        </Form>
      </Drawer>
    </Card>
  );
};

export default CharityTypesAdminPage;
