import { useEffect, useMemo, useState } from "react";
import { Card, Table, Tag, Button, Drawer, Form, Select, Input, InputNumber, message, Space, Typography, Row, Col, Alert, Modal } from "antd";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../../api/client";
import { downloadCsv } from "../../utils/exportCsv";
import { useSearchParams } from "react-router-dom";

const { Text } = Typography;

const LoanRequestsAdminPage = () => {
  const queryClient = useQueryClient();
  const [searchParams, setSearchParams] = useSearchParams();
  const focusId = searchParams.get("focus");
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [overrideEdit, setOverrideEdit] = useState(false);
  const [form] = Form.useForm();
  const [search, setSearch] = useState("");
  const [statusFilter, setStatusFilter] = useState<"pending" | "approved" | "released" | "rejected" | undefined>();

  const { data, isLoading } = useQuery({
    queryKey: ["loan-requests-admin"],
    queryFn: async () => {
      const response = await api.get("/loan-requests");
      return response.data.data;
    }
  });

  const { data: cashiers } = useQuery({
    queryKey: ["cashiers"],
    queryFn: async () => {
      const response = await api.get("/cashiers");
      return response.data.data;
    }
  });

  const { data: cashierSummary } = useQuery({
    queryKey: ["cashier-summary"],
    queryFn: async () => {
      const response = await api.get("/reports/cashier-summary");
      return response.data.data;
    }
  });

  const cashierBalanceMap = useMemo(() => {
    const map = new Map<number, number>();
    (cashierSummary || []).forEach((row: any) => {
      if (typeof row.id === "number") map.set(row.id, Number(row.balance || 0));
    });
    return map;
  }, [cashierSummary]);

  const { data: memberships } = useQuery({
    queryKey: ["memberships"],
    queryFn: async () => {
      const response = await api.get("/memberships");
      return response.data.data;
    }
  });

  const memberNameMap = useMemo(() => {
    const map = new Map<string, string>();
    (memberships || []).forEach((m: any) => {
      map.set(m.member_id, m.member_short_name);
    });
    return map;
  }, [memberships]);

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    return (data || []).filter((row: any) => {
      const memberLabel = `${row.member_id} ${memberNameMap.get(row.member_id) || ""}`.toLowerCase();
      if (statusFilter && row.status !== statusFilter) return false;
      if (!q) return true;
      return (
        String(row.request_id || "").toLowerCase().includes(q) ||
        memberLabel.includes(q) ||
        String(row.status || "").toLowerCase().includes(q)
      );
    });
  }, [data, search, memberNameMap, statusFilter]);

  useEffect(() => {
    if (!focusId) return;
    const record = (data || []).find((r: any) => String(r.request_id) === String(focusId));
    if (!record) return;
    setStatusFilter(undefined);
    setSearch("");
    setTimeout(() => onEdit(record), 0);
    setSearchParams((prev) => {
      prev.delete("focus");
      return prev;
    });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [focusId, data]);

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    setOverrideEdit(false);
    form.setFieldsValue({
      status: record.status,
      allocations: record.cashier_id
        ? [{ cashier_id: record.cashier_id, amount: Number(record.amount || 0) }]
        : [{ cashier_id: undefined, amount: Number(record.amount || 0) }]
    });
  };

  const onSave = async (values: any) => {
    try {
      const status = values.status;
      const payload: any = { status };
      if (status === "released") {
        if (editing.status !== "approved") {
          message.error("Only an approved request can be released.");
          return;
        }
        const allocations = (values.allocations || []).filter((a: any) => a && a.cashier_id && a.amount);
        const total = allocations.reduce((sum: number, a: any) => sum + Number(a.amount || 0), 0);
        if (Math.abs(Number(total.toFixed(2)) - Number(Number(editing.amount || 0).toFixed(2))) > 0.01) {
          message.error("Allocation total must equal the requested amount.");
          return;
        }
        for (const a of allocations) {
          const bal = cashierBalanceMap.get(Number(a.cashier_id)) ?? 0;
          if (Number(a.amount) > Number(bal) + 1e-9) {
            message.error("One or more allocations exceed cashier balance.");
            return;
          }
        }
        payload.allocations = allocations.map((a: any) => ({ cashier_id: Number(a.cashier_id), amount: Number(a.amount) }));
        payload.cashier_id = payload.allocations[0]?.cashier_id;
      }

      if (editing.status === "released" && overrideEdit) {
        payload.force = true;
      }

      await api.put(`/loan-requests/${editing.request_id}`, payload);
      message.success("Loan request updated");
      setOpen(false);
      setEditing(null);
      form.resetFields();
      queryClient.invalidateQueries({ queryKey: ["loan-requests-admin"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Update failed");
    }
  };

  return (
    <Card
      className="panel"
      title="Loan Requests"
      extra={
        <Space>
          <Select
            allowClear
            placeholder="Status"
            style={{ width: 160 }}
            value={statusFilter}
            onChange={(v) => setStatusFilter(v)}
            options={[
              { label: "Pending", value: "pending" },
              { label: "Approved", value: "approved" },
              { label: "Released", value: "released" },
              { label: "Rejected", value: "rejected" }
            ]}
          />
          <Input.Search placeholder="Search" allowClear onChange={(e) => setSearch(e.target.value)} style={{ width: 220 }} />
          <Button
            onClick={() =>
              downloadCsv("loan_requests.csv", filtered, [
                { title: "Request ID", dataIndex: "request_id" },
                { title: "Member ID", dataIndex: "member_id" },
                { title: "Amount", dataIndex: "amount" },
                { title: "Term Months", dataIndex: "term_months" },
                { title: "Status", dataIndex: "status" }
              ])
            }
          >
            Export CSV
          </Button>
        </Space>
      }
    >
      <Table
        loading={isLoading}
        dataSource={filtered}
        rowKey="id"
        columns={[
          { title: "Request ID", dataIndex: "request_id" },
          {
            title: "Member",
            dataIndex: "member_id",
            render: (value: string) => `${value} ${memberNameMap.get(value) ? `- ${memberNameMap.get(value)}` : ""}`
          },
          { title: "Amount", dataIndex: "amount" },
          {
            title: "Status",
            dataIndex: "status",
            render: (value) => (
              <Tag
                color={
                  value === "released" ? "blue" : value === "approved" ? "green" : value === "rejected" ? "red" : "gold"
                }
              >
                {value}
              </Tag>
            )
          },
          {
            title: "Actions",
            render: (_: any, record: any) => (
              <Space>
                <Button size="small" onClick={() => onEdit(record)}>
                  Review
                </Button>
              </Space>
            )
          }
        ]}
      />

      <Drawer
        title={editing ? `Loan Request ${editing.request_id}` : "Loan Request"}
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={680}
      >
        {editing && (
          <>
            {editing.status === "released" && !overrideEdit && (
              <Alert
                type="warning"
                showIcon
                message="This request is already released."
                description="Editing released requests can cause inconsistencies. Use override only if you know what you’re doing."
                action={
                  <Button
                    danger
                    onClick={() => {
                      Modal.confirm({
                        title: "Override and edit released request?",
                        content: "This will allow editing. Financial postings are not re-created automatically.",
                        okText: "Override",
                        okButtonProps: { danger: true },
                        cancelText: "Cancel",
                        onOk: () => setOverrideEdit(true)
                      });
                    }}
                  >
                    Override & Edit
                  </Button>
                }
                style={{ marginBottom: 12 }}
              />
            )}
            <Form form={form} layout="vertical" onFinish={onSave} disabled={editing.status === "released" && !overrideEdit}>
            <Form.Item label="Request ID">
              <Input value={editing.request_id} readOnly />
            </Form.Item>
            <Form.Item label="Member ID">
              <Input value={editing.member_id} readOnly />
            </Form.Item>
            <Form.Item label="Amount">
              <InputNumber value={editing.amount} readOnly style={{ width: "100%" }} />
            </Form.Item>
            <Form.Item label="Term (months)">
              <InputNumber value={editing.term_months} readOnly style={{ width: "100%" }} />
            </Form.Item>
            <Form.Item label="Reason">
              <Input.TextArea value={editing.reason} readOnly rows={3} />
            </Form.Item>
            <Form.Item name="status" label="Status" rules={[{ required: true }]}>
              <Select
                options={[
                  { label: "Pending", value: "pending" },
                  { label: "Approved", value: "approved" },
                  { label: "Released", value: "released" },
                  { label: "Rejected", value: "rejected" }
                ]}
              />
            </Form.Item>

            <Form.Item shouldUpdate noStyle>
              {() => {
                const status = form.getFieldValue("status");
                if (status !== "released") return null;

                const allocations = form.getFieldValue("allocations") || [];
                const total = allocations.reduce((sum: number, a: any) => sum + Number(a?.amount || 0), 0);
                const remaining = Number(editing.amount || 0) - total;
                const selectedCashiers = new Set<number>(
                  allocations.map((a: any) => Number(a?.cashier_id)).filter((v: any) => Number.isFinite(v))
                );

                return (
                  <Card className="panel" style={{ marginBottom: 12 }} title="Disbursement Allocation">
                    <Form.List name="allocations">
                      {(fields, { add, remove }) => (
                        <Space direction="vertical" size={10} style={{ width: "100%" }}>
                          {fields.map((field) => {
                            const rowCashierId = form.getFieldValue(["allocations", field.name, "cashier_id"]);
                            const rowBalance = rowCashierId ? cashierBalanceMap.get(Number(rowCashierId)) ?? 0 : 0;
                            return (
                              <Row key={field.key} gutter={[8, 8]} style={{ width: "100%" }}>
                                <Col xs={24} sm={14}>
                                  <Form.Item
                                    {...field}
                                    name={[field.name, "cashier_id"]}
                                    rules={[{ required: true, message: "Cashier required" }]}
                                    style={{ marginBottom: 0 }}
                                  >
                                    <Select
                                      placeholder="Cashier"
                                      showSearch
                                      optionFilterProp="label"
                                      options={(cashiers || [])
                                        .filter((c: any) => !selectedCashiers.has(Number(c.id)) || Number(c.id) === Number(rowCashierId))
                                        .map((c: any) => ({
                                          label: `${c.name} (Bal: ${(cashierBalanceMap.get(Number(c.id)) ?? 0).toFixed(2)})`,
                                          value: c.id
                                        }))}
                                    />
                                  </Form.Item>
                                </Col>
                                <Col xs={12} sm={6}>
                                  <Form.Item
                                    {...field}
                                    name={[field.name, "amount"]}
                                    rules={[{ required: true, message: "Amount required" }]}
                                    style={{ marginBottom: 0 }}
                                  >
                                    <InputNumber min={0.01} step={0.01} style={{ width: "100%" }} />
                                  </Form.Item>
                                </Col>
                                <Col xs={12} sm={4}>
                                  <Button danger block onClick={() => remove(field.name)}>
                                    Remove
                                  </Button>
                                  <div style={{ marginTop: 6 }}>
                                    <Text type="secondary">Bal: {Number(rowBalance).toFixed(2)}</Text>
                                  </div>
                                </Col>
                              </Row>
                            );
                          })}
                          <Button onClick={() => add({ amount: Math.max(0, remaining) })} disabled={remaining <= 0}>
                            Add Cashier
                          </Button>
                          <Text type={remaining === 0 ? "success" : remaining < 0 ? "danger" : "warning"}>
                            Remaining: {remaining.toFixed(2)}
                          </Text>
                        </Space>
                      )}
                    </Form.List>
                  </Card>
                );
              }}
            </Form.Item>

            <Button type="primary" htmlType="submit" block>
              Update
            </Button>
          </Form>
          </>
        )}
      </Drawer>
    </Card>
  );
};

export default LoanRequestsAdminPage;
