import { Card, Row, Col, Button, Statistic, List, Typography, Space } from "antd";
import { useQuery } from "@tanstack/react-query";
import api from "../../api/client";
import { useNavigate } from "react-router-dom";
import { useAuthStore } from "../../store/authStore";
import { useMemo } from "react";

const { Text } = Typography;

const AdminOverviewPage = () => {
  const navigate = useNavigate();
  const user = useAuthStore((s) => s.user);
  const isAdmin = user?.access_level === "administrator";

  const { data } = useQuery({
    queryKey: ["admin-dashboard-summary"],
    queryFn: async () => {
      const response = await api.get("/admin/dashboard-summary");
      return response.data.data;
    }
  });

  const { data: loanRequests } = useQuery({
    queryKey: ["admin-loan-requests"],
    queryFn: async () => {
      const response = await api.get("/loan-requests");
      return response.data.data;
    },
    enabled: isAdmin
  });

  const { data: charityRequests } = useQuery({
    queryKey: ["admin-charity-requests"],
    queryFn: async () => {
      const response = await api.get("/charity-requests");
      return response.data.data;
    },
    enabled: isAdmin
  });

  const { data: releaseRequests } = useQuery({
    queryKey: ["admin-release-requests"],
    queryFn: async () => {
      const response = await api.get("/release-requests");
      return response.data.data;
    },
    enabled: isAdmin
  });

  const { data: memberships } = useQuery({
    queryKey: ["memberships-admin-overview"],
    queryFn: async () => (await api.get("/memberships")).data.data,
    enabled: isAdmin
  });

  const memberNameMap = useMemo(() => {
    const map = new Map<string, string>();
    (memberships || []).forEach((m: any) => map.set(m.member_id, m.member_short_name));
    return map;
  }, [memberships]);

  const pendingLoans = (data?.pending_loans || 0) as number;
  const pendingCharity = (data?.pending_charity || 0) as number;
  const pendingRelease = (data?.pending_release || 0) as number;

  const pendingLoanList = useMemo(
    () => (loanRequests || []).filter((r: any) => r.status === "pending").slice(0, 6),
    [loanRequests]
  );
  const pendingCharityList = useMemo(
    () => (charityRequests || []).filter((r: any) => r.status === "pending").slice(0, 6),
    [charityRequests]
  );
  const pendingReleaseList = useMemo(
    () => (releaseRequests || []).filter((r: any) => r.status === "pending").slice(0, 6),
    [releaseRequests]
  );

  return (
    <div>
      <div className="stat-strip">
        <div className="stat-tile">
          <Statistic title="Total Users" value={data?.total_users || 0} />
        </div>
        <div className="stat-tile">
          <Statistic title="Active Members" value={data?.active_members || 0} />
        </div>
        <div className="stat-tile">
          <Statistic title="Pending Loans" value={pendingLoans} />
          <Button type="link" style={{ padding: 0 }} onClick={() => navigate("/app/admin/loan-requests")}>
            View
          </Button>
        </div>
        <div className="stat-tile">
          <Statistic title="Pending Charity" value={pendingCharity} />
          <Button type="link" style={{ padding: 0 }} onClick={() => navigate("/app/admin/charity-requests")}>
            View
          </Button>
        </div>
        <div className="stat-tile">
          <Statistic title="Pending Releases" value={pendingRelease} />
          <Button type="link" style={{ padding: 0 }} onClick={() => navigate("/app/admin/release-requests")}>
            View
          </Button>
        </div>
      </div>

      <div style={{ marginTop: 16 }}>
        <Row gutter={[16, 16]}>
          <Col xs={24} md={8}>
            <Card className="panel" title="Loan Requests">
              <div style={{ display: "flex", justifyContent: "space-between", alignItems: "center" }}>
                <div style={{ color: "var(--app-muted)" }}>Pending</div>
                <div style={{ fontSize: 22, fontWeight: 700 }}>{pendingLoans}</div>
              </div>
              <Button
                type="primary"
                block
                style={{ marginTop: 10 }}
                onClick={() => navigate("/app/admin/loan-requests")}
                disabled={!isAdmin}
              >
                View All
              </Button>
              {isAdmin && (
                <div style={{ marginTop: 10 }}>
                  <List
                    size="small"
                    dataSource={pendingLoanList}
                    locale={{ emptyText: <Text type="secondary">No pending records</Text> }}
                    renderItem={(item: any) => (
                      <List.Item
                        style={{ paddingInline: 0 }}
                        actions={[
                          <Button
                            key="open"
                            size="small"
                            type="link"
                            onClick={() => navigate(`/app/admin/loan-requests?focus=${encodeURIComponent(item.request_id)}`)}
                          >
                            View
                          </Button>
                        ]}
                      >
                        <Text strong>
                          {item.request_id} — {item.member_id}
                          {memberNameMap.get(item.member_id) ? ` (${memberNameMap.get(item.member_id)})` : ""}
                        </Text>
                      </List.Item>
                    )}
                  />
                </div>
              )}
            </Card>
          </Col>
          <Col xs={24} md={8}>
            <Card className="panel" title="Charity Requests">
              <div style={{ display: "flex", justifyContent: "space-between", alignItems: "center" }}>
                <div style={{ color: "var(--app-muted)" }}>Pending</div>
                <div style={{ fontSize: 22, fontWeight: 700 }}>{pendingCharity}</div>
              </div>
              <Button
                type="primary"
                block
                style={{ marginTop: 10 }}
                onClick={() => navigate("/app/admin/charity-requests")}
                disabled={!isAdmin}
              >
                View All
              </Button>
              {isAdmin && (
                <div style={{ marginTop: 10 }}>
                  <List
                    size="small"
                    dataSource={pendingCharityList}
                    locale={{ emptyText: <Text type="secondary">No pending records</Text> }}
                    renderItem={(item: any) => (
                      <List.Item
                        style={{ paddingInline: 0 }}
                        actions={[
                          <Button
                            key="open"
                            size="small"
                            type="link"
                            onClick={() => navigate(`/app/admin/charity-requests?focus=${encodeURIComponent(item.charity_id)}`)}
                          >
                            View
                          </Button>
                        ]}
                      >
                        <Text strong>
                          {item.charity_id} — {item.member_id}
                          {memberNameMap.get(item.member_id) ? ` (${memberNameMap.get(item.member_id)})` : ""}
                        </Text>
                      </List.Item>
                    )}
                  />
                </div>
              )}
            </Card>
          </Col>
          <Col xs={24} md={8}>
            <Card className="panel" title="Release Requests">
              <div style={{ display: "flex", justifyContent: "space-between", alignItems: "center" }}>
                <div style={{ color: "var(--app-muted)" }}>Pending</div>
                <div style={{ fontSize: 22, fontWeight: 700 }}>{pendingRelease}</div>
              </div>
              <Button
                type="primary"
                block
                style={{ marginTop: 10 }}
                onClick={() => navigate("/app/admin/release-requests")}
                disabled={!isAdmin}
              >
                View All
              </Button>
              {isAdmin && (
                <div style={{ marginTop: 10 }}>
                  <List
                    size="small"
                    dataSource={pendingReleaseList}
                    locale={{ emptyText: <Text type="secondary">No pending records</Text> }}
                    renderItem={(item: any) => (
                      <List.Item
                        style={{ paddingInline: 0 }}
                        actions={[
                          <Button
                            key="open"
                            size="small"
                            type="link"
                            onClick={() =>
                              navigate(`/app/admin/release-requests?focus=${encodeURIComponent(item.release_request_id)}`)
                            }
                          >
                            View
                          </Button>
                        ]}
                      >
                        <Text strong>
                          {item.release_request_id} — {item.member_id}
                          {memberNameMap.get(item.member_id) ? ` (${memberNameMap.get(item.member_id)})` : ""}
                        </Text>
                      </List.Item>
                    )}
                  />
                </div>
              )}
            </Card>
          </Col>
        </Row>
      </div>
    </div>
  );
};

export default AdminOverviewPage;
